/**
 * CyTube Playlist Importer
 * Adds functionality to import playlists from external URLs (TXT format only)
 * Parses the playlist and adds all media URLs to the CyTube queue
 */

(function() {
    'use strict';

    // Wait for CyTube to be fully loaded
    if (typeof socket === 'undefined' || typeof $ === 'undefined') {
        console.error('[Playlist Importer] CyTube socket or jQuery not available yet');
        setTimeout(arguments.callee, 1000);
        return;
    }

    console.log('[Playlist Importer] Initializing...');

    // Configuration
    var CONFIG = {
        // Delay between queueing each video (milliseconds) - prevents rate limiting
        queueDelay: 1000,
        // Maximum number of items to queue from a single playlist
        maxItems: 500
    };

    /**
     * Parse plain text format (one URL per line)
     */
    function parsePlainText(content) {
        var lines = content.split('\n');
        var items = [];

        lines.forEach(function(line) {
            line = line.trim();
            if (line && line.match(/^https?:\/\//i)) {
                items.push({
                    url: line,
                    title: null
                });
            }
        });

        return items;
    }

    /**
     * Parse playlist (plain text only)
     */
    function parsePlaylist(content) {
        return parsePlainText(content);
    }

    /**
     * Fetch playlist from URL using CORS proxy if needed
     */
    function fetchPlaylist(url, callback) {
        // Try direct fetch first
        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'text',
            timeout: 10000,
            success: function(data) {
                callback(null, data);
            },
            error: function(xhr, status, error) {
                // If CORS error, try with a CORS proxy
                var corsProxies = [
                    'https://corsproxy.io/?',
                    'https://api.allorigins.win/raw?url='
                ];

                var proxyUrl = corsProxies[0] + encodeURIComponent(url);
                $.ajax({
                    url: proxyUrl,
                    type: 'GET',
                    dataType: 'text',
                    timeout: 10000,
                    success: function(data) {
                        callback(null, data);
                    },
                    error: function() {
                        callback('Failed to fetch playlist. Try downloading and hosting it yourself, or check CORS settings.');
                    }
                });
            }
        });
    }

    /**
     * Parse media link using CyTube's built-in parser
     */
    function parseMediaLink(url) {
        if (typeof window.parseMediaLink === 'function') {
            return window.parseMediaLink(url);
        }

        // Fallback: assume it's a raw file
        return {
            id: url,
            type: 'fi'
        };
    }

    /**
     * Queue items to CyTube playlist
     */
    function queueItems(items, position, temporary) {
        if (items.length === 0) {
            return;
        }

        if (items.length > CONFIG.maxItems) {
            if (!confirm('This playlist contains ' + items.length + ' items. Only the first ' + CONFIG.maxItems + ' will be added. Continue?')) {
                return;
            }
            items = items.slice(0, CONFIG.maxItems);
        }

        // If adding to "next", reverse the order
        if (position === 'next') {
            items = items.reverse();
        }

        // Queue items with delay to prevent rate limiting
        function queueNext(index) {
            if (index >= items.length) {
                return;
            }

            var item = items[index];
            var mediaData;

            try {
                mediaData = parseMediaLink(item.url);
            } catch (error) {
                console.error('[Playlist Importer] Failed to parse:', item.url, error);
                queueNext(index + 1);
                return;
            }

            if (!mediaData || !mediaData.id || !mediaData.type) {
                console.error('[Playlist Importer] Invalid media data for:', item.url);
                queueNext(index + 1);
                return;
            }

            var queueData = {
                id: mediaData.id,
                type: mediaData.type,
                pos: position,
                temp: temporary,
                title: item.title || undefined
            };

            socket.emit('queue', queueData);

            setTimeout(function() {
                queueNext(index + 1);
            }, CONFIG.queueDelay);
        }

        queueNext(0);
    }

    /**
     * Show notification message
     */
    function showNotification(message, type) {
        var alertClass = 'alert-' + (type || 'info');
        var notification = $('#playlist-importer-status');
        notification.html('<div class="alert ' + alertClass + '">' + message + '</div>');
        setTimeout(function() {
            notification.empty();
        }, 5000);
    }

    /**
     * Handle playlist import
     */
    function handleImport(position) {
        var playlistUrl = $('#playlist-url').val().trim();

        if (!playlistUrl) {
            showNotification('Please enter a playlist URL', 'warning');
            return;
        }

        if (!playlistUrl.match(/^https?:\/\//i)) {
            showNotification('Please enter a valid URL starting with http:// or https://', 'warning');
            return;
        }

        fetchPlaylist(playlistUrl, function(error, content) {
            if (error) {
                showNotification(error, 'danger');
                return;
            }

            var items = parsePlaylist(content);

            if (items.length === 0) {
                showNotification('No media URLs found in playlist', 'warning');
                return;
            }

            var temporary = $('#playlist-importer-temp').prop('checked');
            queueItems(items, position, temporary);
        });
    }

    /**
     * Create and inject UI elements
     */
    function createUI() {
        // Add button to playlist controls
        var button = $('<button>')
            .attr('id', 'showplaylistimporter')
            .attr('title', 'Import playlist from URL')
            .attr('data-toggle', 'collapse')
            .attr('data-target', '#playlistimporter')
            .addClass('btn btn-sm btn-default')
            .html('<span class="glyphicon glyphicon-import"></span>')
            .on('click', function() {
                // Close other panels
                $('#addfromurl, #searchcontrol, #customembed, #playlistmanager').collapse('hide');
            });

        // Insert button after the "show media url" button
        $('#showmediaurl').after(button);

        // Create collapsible panel
        var panel = $('<div>')
            .attr('id', 'playlistimporter')
            .addClass('collapse plcontrol-collapse col-lg-12 col-md-12')
            .html(
                '<div class="vertical-spacer"></div>' +
                '<div class="input-group">' +
                    '<input id="playlist-url" class="form-control" type="text" placeholder="Playlist URL (TXT - one URL per line)">' +
                    '<span class="input-group-btn">' +
                        '<button id="playlist-queue-next" class="btn btn-default">Queue next</button>' +
                    '</span>' +
                    '<span class="input-group-btn">' +
                        '<button id="playlist-queue-end" class="btn btn-default">Queue last</button>' +
                    '</span>' +
                '</div>' +
                '<div class="checkbox">' +
                    '<label>' +
                        '<input id="playlist-importer-temp" type="checkbox" checked> Add as temporary' +
                    '</label>' +
                '</div>' +
                '<div id="playlist-importer-status"></div>'
            );

        // Insert panel after search control
        $('#searchcontrol').after(panel);

        // Attach event handlers
        $('#playlist-queue-next').on('click', function() {
            handleImport('next');
        });

        $('#playlist-queue-end').on('click', function() {
            handleImport('end');
        });

        $('#playlist-url').on('keypress', function(e) {
            if (e.which === 13) {
                handleImport('end');
            }
        });

        console.log('[Playlist Importer] UI created successfully');
    }

    /**
     * Initialize the addon
     */
    function init() {
        // Wait for the page to be fully loaded
        if ($('#showmediaurl').length === 0) {
            setTimeout(init, 500);
            return;
        }

        createUI();
        console.log('[Playlist Importer] Initialized successfully');
    }

    // Start initialization
    init();

})();
